	/*
CREATE DATABASE 'C:\projects\db\test' USER 'SYSDBA' PASSWORD 'mypassword'
       PAGE_SIZE = 16384
       DEFAULT CHARACTER SET UTF8 COLLATION UNICODE_CI_AI;
*/

/*
 * emulate boolean type
 */
CREATE DOMAIN BoolType 
    AS CHAR(1)
    CHECK (VALUE IS NULL OR VALUE IN (0,1));

CREATE TABLE Application 
(
	Ident SMALLINT NOT NULL,
	Name  VARCHAR(30) NOT NULL,
	Desc VARCHAR(1000),
        CONSTRAINT App_Ident_is_PKey PRIMARY KEY (Ident),
        CONSTRAINT App_Name_is_Not_Empty CHECK (Name <> ''),
        CONSTRAINT App_Name_is_NoSpaces CHECK (CHAR_LENGTH(Name) = CHAR_LENGTH(TRIM(Name))),
        CONSTRAINT App_Name_is_Unique UNIQUE (Name)
);
COMMIT;

/*
 * Possible user roles:
 * Guest        = 1
 * User         = 2
 * UserRealTime = 3
 * Expert       = 4
 * RealTimer    = 5  
 */
CREATE TABLE UserRole
(
    Ident  SMALLINT NOT NULL,
    Name   VARCHAR(20) NOT NULL,
        CONSTRAINT UserRole_Ident_is_PKey PRIMARY KEY (Ident),
        CONSTRAINT UserRole_Name_is_Not_Empty CHECK (Name <> ''),
        CONSTRAINT UserRole_Name_is_NoSpaces CHECK (CHAR_LENGTH(Name) = CHAR_LENGTH(TRIM(Name))),
        CONSTRAINT UserRole_Name_is_Unique UNIQUE (Name)
);
COMMIT;

CREATE TABLE Content 
(
	Ident	SMALLINT NOT NULL,
	Name	VARCHAR(64)	NOT NULL,
	Desc	VARCHAR(1000),
        CONSTRAINT Content_Ident_is_PKey PRIMARY KEY (Ident),
        CONSTRAINT Content_Name_is_Not_Empty CHECK (Name <> ''),
        CONSTRAINT Content_Name_is_NoSpaces CHECK (CHAR_LENGTH(Name) = CHAR_LENGTH(TRIM(Name))),
        CONSTRAINT Content_Name_is_Unique UNIQUE (Name)
);
COMMIT;

CREATE TABLE Database
(
    Ident	SMALLINT NOT NULL,
    DbTitle	VARCHAR(512) NOT NULL,
    Protocol VARCHAR(100) NOT NULL,
    Host	VARCHAR(100) NOT NULL,
    DbFile	VARCHAR(512) NOT NULL,
    Driver	VARCHAR(100) NOT NULL COLLATE UNICODE,
        CONSTRAINT DbDesc_Ident_is_PKey PRIMARY KEY (Ident),
        CONSTRAINT DbDesc_DbTitle_is_Not_Empty CHECK (DbTitle <> ''),
        CONSTRAINT DbDesc_DbTitle_is_NoSpaces CHECK (CHAR_LENGTH(DbTitle) = CHAR_LENGTH(TRIM(DbTitle))),
        CONSTRAINT DbDesc_DbTitle_is_Unique UNIQUE (DbTitle),
        CONSTRAINT DbDesc_Protocol_is_Not_Empty CHECK (Protocol <> ''),
        CONSTRAINT DbDesc_Protocol_is_NoSpaces CHECK (CHAR_LENGTH(Protocol) = CHAR_LENGTH(TRIM(Protocol))),
        CONSTRAINT DbDesc_Host_is_Not_Empty CHECK (Host <> ''),
        CONSTRAINT DbDesc_Host_is_NoSpaces CHECK (CHAR_LENGTH(Host) = CHAR_LENGTH(TRIM(Host))),
        CONSTRAINT DbDesc_DbFile_is_Not_Empty CHECK (DbFile <> ''),
        CONSTRAINT DbDesc_DbFile_is_NoSpaces CHECK (CHAR_LENGTH(DbFile) = CHAR_LENGTH(TRIM(DbFile))),
        CONSTRAINT DbDesc_Driver_is_Not_Empty CHECK (Driver <> ''),
        CONSTRAINT DbDesc_Driver_is_NoSpaces CHECK (CHAR_LENGTH(Driver) = CHAR_LENGTH(TRIM(Driver)))
);
COMMIT;

CREATE TABLE ContentUsage
(
	AppId	SMALLINT NOT NULL,
	UserRoleId	SMALLINT NOT NULL,
	ContentId	SMALLINT NOT NULL,
        CONSTRAINT ContUsage_AppRoleCont_is_PKey PRIMARY KEY (AppId, UserRoleId, ContentId),
        CONSTRAINT ContUsage_AppId_is_FKey FOREIGN KEY (AppId) REFERENCES Application ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT ContUsage_UserRoleId_is_FKey FOREIGN KEY (UserRoleId) REFERENCES UserRole ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT ContUsage_ContentId_is_FKey FOREIGN KEY (ContentId) REFERENCES Content ON DELETE CASCADE ON UPDATE CASCADE
);
COMMIT;

CREATE TABLE ContentLocation
(
	ContentId	SMALLINT NOT NULL,
	DbId		SMALLINT NOT NULL,
        CONSTRAINT ContentLoc_ContIdDbId_is_PKey PRIMARY KEY (ContentId, DbId),
        CONSTRAINT ContentLoc_ContentId_is_FKey FOREIGN KEY (ContentId) REFERENCES Content ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT ContentLoc_DbId_is_FKey FOREIGN KEY (DbId) REFERENCES Database ON DELETE CASCADE ON UPDATE CASCADE
);
COMMIT;

CREATE TABLE DbCredential
(
    AppId	INT NOT NULL,
    UserRoleId    INT NOT NULL,
    DbId	SMALLINT NOT NULL,
    DbUser  VARCHAR(20) NOT NULL,
    DbPass	VARCHAR(50) NOT NULL COLLATE UNICODE,
    DbRole	VARCHAR(20),
        CONSTRAINT DbCred_AppIdRoleIdDbId_is_PKey PRIMARY KEY (AppId, UserRoleId, DbId),
        CONSTRAINT DbCred_AppId_is_FKey FOREIGN KEY (AppId) REFERENCES Application ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT DbCred_UserRoleId_is_FKey FOREIGN KEY (UserRoleId) REFERENCES UserRole ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT DbCred_DbId_is_FKey FOREIGN KEY (DbId) REFERENCES Database ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT DbCred_DbUser_is_Not_Empty CHECK (DbUser <> ''),
        CONSTRAINT DbCred_DbUser_is_NoSpaces CHECK (CHAR_LENGTH(DbUser) = CHAR_LENGTH(TRIM(DbUser))),
        CONSTRAINT DbCred_DbPass_is_Not_Empty CHECK (DbPass <> ''),
        CONSTRAINT DbCred_DbPass_is_NoSpaces CHECK (CHAR_LENGTH(DbPass) = CHAR_LENGTH(TRIM(DbPass))),
        CONSTRAINT DbCred_DbRole_is_Not_Empty CHECK (DbRole <> ''),
        CONSTRAINT DbCred_DbRole_is_NoSpaces CHECK (CHAR_LENGTH(DbRole) = CHAR_LENGTH(TRIM(DbRole)))
);
COMMIT;

/***********************************************************
 Table: User, Owner SYSDBA
   Web Applications' Users table
 **********************************************************/
CREATE TABLE AppUser
(
    /* Unique Generated Identification Number for internal use only */
    Ident       INT NOT NULL,
    /* User log Name, unique */
    Name        VARCHAR(20) NOT NULL,
    /* 0 - account is not locked; 1 - account is locked, any user activity is prohibited */ 
    Locked      BoolType DEFAULT 0 NOT NULL,
    /* 
     * Temporary password.
     * If temporary password is not expired then the only activity whcich is allowed is setting new password. After setting new (not-temporary) 
     * password user have to register again using new password to get normal access to applications.
     *
     * Temporary password is expired if TmpPassTime is ealier than NOW - system-defined-resset-password time period
     * 
     * NOTE:
     * if this field is NULL user registration process uses its normal password, HashedPass.
     */
    TmpPass     VARCHAR(10) COLLATE UNICODE,
    /*
     * UT Time of temporary password setup.
     * NOTE:
     * this field is NOT NULL if TmpPass IS NOT NULL
     */
    TmpPassTime Timestamp,
    /* User password */
    HashedPass  BLOB NOT NULL,
    /* Salt to increase crypt security of hashed salted password */
    Salt        BLOB NOT NULL,
    /* E-mail of person or organization's representative  */
    Email       VARCHAR(50) NOT NULL,
    /* Person name responsible for account */
    ContactName	VARCHAR(50) NOT NULL,       
    /* Organization */
    Organization 	VARCHAR(200) NOT NULL,       
        CONSTRAINT AppUser_Ident_is_PKey PRIMARY KEY (Ident),
        CONSTRAINT AppUser_Name_is_Not_Empty CHECK (Name <> ''),
        CONSTRAINT AppUser_Name_is_NoSpaces CHECK (CHAR_LENGTH(Name) = CHAR_LENGTH(TRIM(Name))),
        CONSTRAINT AppUser_Name_is_Unique UNIQUE (Name),
        CONSTRAINT AppUser_Email_is_Not_Empty CHECK (Email <> ''),
        CONSTRAINT AppUser_Email_is_NoSpaces CHECK (CHAR_LENGTH(Email) = CHAR_LENGTH(TRIM(Email))),
        CONSTRAINT AppUser_ContactName_isNotEmpty CHECK (ContactName <> ''),
        CONSTRAINT AppUser_ContactName_is_NoSpaces CHECK (CHAR_LENGTH(ContactName) = CHAR_LENGTH(TRIM(ContactName))),
        CONSTRAINT AppUser_Org_is_Not_Empty CHECK (Organization <> ''),
        CONSTRAINT AppUser_Org_is_NoSpaces CHECK (CHAR_LENGTH(Organization) = CHAR_LENGTH(TRIM(Organization)))
);
COMMIT;

CREATE TABLE AppGuest
(
    /* Unique Generated Identification Number for internal use only */
    Ident       INT NOT NULL,
    /* E-mail of person or organization's representative  */
    Email       VARCHAR(50) NOT NULL,
    /* Organization */
    Organization 	VARCHAR(200) NOT NULL,       
        CONSTRAINT AppGuest_Ident_is_PKey PRIMARY KEY (Ident),
        CONSTRAINT AppGuest_Email_is_Not_Empty CHECK (Email <> ''),
        CONSTRAINT AppGuest_Email_is_NoSpaces CHECK (CHAR_LENGTH(Email) = CHAR_LENGTH(TRIM(Email))),
        CONSTRAINT AppGuest_Org_is_Not_Empty CHECK (Organization <> ''),
        CONSTRAINT AppGuest_Org_is_NoSpaces CHECK (CHAR_LENGTH(Organization) = CHAR_LENGTH(TRIM(Organization))),
        CONSTRAINT AppGuest_EmailOrg_is_Unique UNIQUE (Email, Organization)
);
COMMIT;

/***********************************************************
 Table: AppExpert
   Category of users with permission to write to the databases
   Can be human or intelligent spftware
 **********************************************************/
CREATE TABLE AppExpert
(
  Ident        SMALLINT NOT NULL,
  LogName      CHAR(10) NOT NULL,
  FName        CHAR(20) NOT NULL,
  LName        CHAR(20) NOT NULL,
  Human        SMALLINT DEFAULT 1 NOT NULL, /* 1 - human, 0 - program */
  Rank         FLOAT NOT NULL,
    CONSTRAINT Scaler_Ident_IS_PRIMARY_KEY PRIMARY KEY( Ident ),
    CONSTRAINT Scaler_LogName_is_unique UNIQUE( LogName ),
    CONSTRAINT Scaler_LogName_is_caps CHECK( UPPER( LogName ) = LogName )
);
COMMIT;


CREATE TABLE Permission
(
    Ident  INT NOT NULL,
    AppId  SMALLINT NOT NULL,
    UserId INT,
    UserRoleId  SMALLINT NOT NULL,
    Settings BLOB SUB_TYPE 1, 
        CONSTRAINT Perm_Ident_is_PKey PRIMARY KEY (Ident),
        CONSTRAINT Perm_AppId_is_FKey FOREIGN KEY (AppId) REFERENCES Application ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT Perm_UserId_is_FKey FOREIGN KEY (UserId) REFERENCES AppUser ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT Perm_RoleId_is_FKey FOREIGN KEY (UserRoleId) REFERENCES UserRole ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT Perm_App_User_is_Unique UNIQUE (AppId, UserId)
);
COMMIT;

CREATE TABLE LogActivity
(
    AppId  SMALLINT NOT NULL,
    UserId INT,
    GuestId INT,
    TimeUT TIMESTAMP NOT NULL,
    IpNumber VARCHAR(50),
        CONSTRAINT LogAct_UserId_is_FKey FOREIGN KEY (UserId) REFERENCES AppUser ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT LogAct_GuestId_is_FKey FOREIGN KEY (GuestId) REFERENCES AppGuest ON DELETE CASCADE ON UPDATE CASCADE,
        CONSTRAINT LogAct_UserOrGuest_is_Not_Null CHECK (UserId IS NULL AND GuestId IS NOT NULL OR UserId IS NOT NULL AND GuestId IS NULL),
        CONSTRAINT LogAct_IpNumber_is_Not_Empty CHECK (IpNumber IS NULL OR IpNumber <> ''),
        CONSTRAINT LogAct_IpNumber_is_NoSpaces CHECK (IpNumber IS NULL OR CHAR_LENGTH(IpNumber) = CHAR_LENGTH(TRIM(IpNumber)))
);
COMMIT;

/* for sequence unique Ident value for new record in Station table */
CREATE SEQUENCE AppUser_Ident_Sequence;
CREATE SEQUENCE AppGuest_Ident_Sequence;
CREATE SEQUENCE Permission_Ident_Sequence;

/* Create various triggers and procedures */

SET TERM !! ;

CREATE TRIGGER Tr_Before_Ins_AppUser FOR AppUser BEFORE INSERT POSITION 0
    AS BEGIN
        NEW.Ident = NEXT VALUE FOR AppUser_Ident_Sequence;
    END!!


CREATE TRIGGER Tr_Before_Ins_AppGuest FOR AppGuest BEFORE INSERT POSITION 0
    AS BEGIN
        NEW.Ident = NEXT VALUE FOR AppGuest_Ident_Sequence;
    END!!

CREATE TRIGGER Tr_Before_Ins_Permission FOR Permission BEFORE INSERT POSITION 0
    AS BEGIN
        NEW.Ident = NEXT VALUE FOR Permission_Ident_Sequence;
    END!!
    
SET TERM ; !!    

/* Create Roles */
CREATE ROLE Registrator; 
CREATE ROLE Administrator;

/* Create Grants */

/* grants for Registrator */
GRANT SELECT ON UserRole     TO Registrator;
GRANT SELECT ON Application  TO Registrator;
GRANT SELECT ON Content      TO Registrator;
GRANT SELECT ON Database     TO Registrator;
GRANT SELECT ON ContentUsage TO Registrator;
GRANT SELECT ON ContentLocation  TO Registrator;
GRANT SELECT ON DbCredential TO Registrator;
GRANT SELECT, INSERT, UPDATE(Locked,TmpPass,tmpPassTime,HashedPass,Salt,Email,ContactName,Organization) ON AppUser TO Registrator;
GRANT SELECT, INSERT ON AppGuest TO Registrator;
GRANT SELECT, INSERT, DELETE, UPDATE(Settings) ON Permission   TO Registrator;
GRANT SELECT, INSERT ON LogActivity  TO Registrator;

/* grants for administrator */
GRANT ALL ON Application  TO Administrator;
GRANT ALL ON UserRole     TO Administrator;
GRANT ALL ON Content      TO Administrator;
GRANT ALL ON Database     TO Administrator;
GRANT ALL ON ContentUsage     TO Administrator;
GRANT ALL ON ContentLocation  TO Administrator;
GRANT ALL ON DbCredential TO Administrator;
GRANT ALL ON AppUser      TO Administrator;
GRANT ALL ON AppGuest     TO Administrator;
GRANT ALL ON Permission   TO Administrator;
GRANT ALL ON LogActivity  TO Administrator;

GRANT Registrator TO lgdc;
GRANT Administrator TO admlgdc;
